/*
 * Copyright 2012 Red Hat Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sub license, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDERS, AUTHORS AND/OR ITS SUPPLIERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE
 * USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 */
/*
 * Authors: Dave Airlie <airlied@redhat.com>
 */
#ifndef __AST_DRV_H__
#define __AST_DRV_H__

#include <linux/i2c.h>
#include <linux/i2c-algo-bit.h>
#include <linux/io.h>
#include <linux/types.h>

#include <drm/drm_connector.h>
#include <drm/drm_crtc.h>
#include <drm/drm_encoder.h>
#include <drm/drm_mode.h>
#include <drm/drm_framebuffer.h>
#include <drm/drm_fb_helper.h>

#include "ast_reg.h"

#define DRIVER_AUTHOR		"Dave Airlie"

#define DRIVER_NAME		"ast"
#define DRIVER_DESC		"AST"
#define DRIVER_DATE	  "20250509"

#define DRIVER_MAJOR	1
#define DRIVER_MINOR	15
#define DRIVER_PATCHLEVEL	2

#define PCI_CHIP_AST2000 0x2000
#define PCI_CHIP_AST2100 0x2010

#define __AST_CHIP(__gen, __index)	((__gen) << 16 | (__index))

enum ast_chip {
	/* 1st gen */
	AST1000 = __AST_CHIP(1, 0), // unused
	AST2000 = __AST_CHIP(1, 1),
	/* 2nd gen */
	AST1100 = __AST_CHIP(2, 0),
	AST2100 = __AST_CHIP(2, 1),
	AST2050 = __AST_CHIP(2, 2), // unused
	/* 3rd gen */
	AST2200 = __AST_CHIP(3, 0),
	AST2150 = __AST_CHIP(3, 1),
	/* 4th gen */
	AST2300 = __AST_CHIP(4, 0),
	AST1300 = __AST_CHIP(4, 1),
	AST1050 = __AST_CHIP(4, 2), // unused
	/* 5th gen */
	AST2400 = __AST_CHIP(5, 0),
	AST1400 = __AST_CHIP(5, 1),
	AST1250 = __AST_CHIP(5, 2), // no VGA
	/* 6th gen */
	AST2500 = __AST_CHIP(6, 0),
	AST2510 = __AST_CHIP(6, 1),
	AST2520 = __AST_CHIP(6, 2), // no VGA
	/* 7th gen */
	AST2600 = __AST_CHIP(7, 0),
	AST2620 = __AST_CHIP(7, 1), // no VGA
};

#define __AST_CHIP_GEN(__chip)	(((unsigned long)(__chip)) >> 16)

enum ast_tx_chip {
	AST_TX_NONE,
	AST_TX_ASTDP,
};

#define AST_DRAM_512Mx16 0
#define AST_DRAM_1Gx16   1
#define AST_DRAM_512Mx32 2
#define AST_DRAM_1Gx32   3
#define AST_DRAM_2Gx16   6
#define AST_DRAM_4Gx16   7
#define AST_DRAM_8Gx16   8

/*
 * Hardware cursor
 */

#define AST_MAX_HWC_WIDTH	64
#define AST_MAX_HWC_HEIGHT	64

#define AST_HWC_SIZE		(AST_MAX_HWC_WIDTH * AST_MAX_HWC_HEIGHT * 2)
#define AST_HWC_SIGNATURE_SIZE	32

/* define for signature structure */
#define AST_HWC_SIGNATURE_CHECKSUM	0x00
#define AST_HWC_SIGNATURE_SizeX		0x04
#define AST_HWC_SIGNATURE_SizeY		0x08
#define AST_HWC_SIGNATURE_X		0x0C
#define AST_HWC_SIGNATURE_Y		0x10
#define AST_HWC_SIGNATURE_HOTSPOTX	0x14
#define AST_HWC_SIGNATURE_HOTSPOTY	0x18

/*
 * Planes
 */

struct ast_plane {
	struct drm_plane base;

	void __iomem *vaddr;
	u64 offset;
	unsigned long size;
};

static inline struct ast_plane *to_ast_plane(struct drm_plane *plane)
{
	return container_of(plane, struct ast_plane, base);
}

/*
 * Connector with i2c channel
 */

struct ast_i2c_chan {
	struct i2c_adapter adapter;
	struct drm_device *dev;
	struct i2c_algo_bit_data bit;
};

struct ast_connector {
	struct drm_connector base;
	struct ast_i2c_chan *i2c;
};

static inline struct ast_connector *
to_ast_connector(struct drm_connector *connector)
{
	return container_of(connector, struct ast_connector, base);
}

/*
 * Device
 */

struct ast_private {
	struct drm_device base;

	void __iomem *regs;
	void __iomem *ioregs;

	enum ast_chip chip;
	uint32_t dram_bus_width;
	uint32_t dram_type;
	uint32_t mclk;

	struct mutex modeset_lock; /* Protects access to modeset I/O registers in ioregs */

	void __iomem	*vram;
	unsigned long	vram_base;
	unsigned long	vram_size;
	unsigned long	vram_fb_available;

	struct ast_plane primary_plane;
	struct ast_plane cursor_plane;
	struct drm_crtc crtc;
	struct drm_encoder encoder;
	struct ast_connector connector;

	bool RefCLK25MHz;
	enum {
		ast_use_p2a,
		ast_use_dt,
		ast_use_defaults
	} config_mode;

	enum ast_tx_chip tx_chip_type;
};

static inline struct ast_private *to_ast_private(struct drm_device *dev)
{
	return container_of(dev, struct ast_private, base);
}

struct ast_private *ast_device_create(const struct drm_driver *drv,
				      struct pci_dev *pdev,
				      unsigned long flags);

static inline unsigned long __ast_gen(struct ast_private *ast)
{
	return __AST_CHIP_GEN(ast->chip);
}
#define AST_GEN(__ast)	__ast_gen(__ast)

static inline bool __ast_gen_is_eq(struct ast_private *ast, unsigned long gen)
{
	return __ast_gen(ast) == gen;
}
#define IS_AST_GEN1(__ast)	__ast_gen_is_eq(__ast, 1)
#define IS_AST_GEN2(__ast)	__ast_gen_is_eq(__ast, 2)
#define IS_AST_GEN3(__ast)	__ast_gen_is_eq(__ast, 3)
#define IS_AST_GEN4(__ast)	__ast_gen_is_eq(__ast, 4)
#define IS_AST_GEN5(__ast)	__ast_gen_is_eq(__ast, 5)
#define IS_AST_GEN6(__ast)	__ast_gen_is_eq(__ast, 6)
#define IS_AST_GEN7(__ast)	__ast_gen_is_eq(__ast, 7)

#define __ast_read(x) \
static inline u##x ast_read##x(struct ast_private *ast, u32 reg) { \
u##x val = 0;\
val = ioread##x(ast->regs + reg); \
return val;\
}

__ast_read(8);
__ast_read(16);
__ast_read(32)

#define __ast_io_read(x) \
static inline u##x ast_io_read##x(struct ast_private *ast, u32 reg) { \
u##x val = 0;\
val = ioread##x(ast->ioregs + reg); \
return val;\
}

__ast_io_read(8);
__ast_io_read(16);
__ast_io_read(32);

#define __ast_write(x) \
static inline void ast_write##x(struct ast_private *ast, u32 reg, u##x val) {\
	iowrite##x(val, ast->regs + reg);\
	}

__ast_write(8);
__ast_write(16);
__ast_write(32);

#define __ast_io_write(x) \
static inline void ast_io_write##x(struct ast_private *ast, u32 reg, u##x val) {\
	iowrite##x(val, ast->ioregs + reg);\
	}

__ast_io_write(8);
__ast_io_write(16);
#undef __ast_io_write

static inline void ast_set_index_reg(struct ast_private *ast,
				     uint32_t base, uint8_t index,
				     uint8_t val)
{
	ast_io_write16(ast, base, ((u16)val << 8) | index);
}

void ast_set_index_reg_mask(struct ast_private *ast,
			    uint32_t base, uint8_t index,
			    uint8_t mask, uint8_t val);
uint8_t ast_get_index_reg(struct ast_private *ast,
			  uint32_t base, uint8_t index);
uint8_t ast_get_index_reg_mask(struct ast_private *ast,
			       uint32_t base, uint8_t index, uint8_t mask);

static inline void ast_open_key(struct ast_private *ast)
{
	ast_set_index_reg(ast, AST_IO_VGACRI, 0x80, 0xA8);
}

#define AST_VIDMEM_SIZE_8M    0x00800000
#define AST_VIDMEM_SIZE_16M   0x01000000
#define AST_VIDMEM_SIZE_32M   0x02000000
#define AST_VIDMEM_SIZE_64M   0x04000000
#define AST_VIDMEM_SIZE_128M  0x08000000

#define AST_VIDMEM_DEFAULT_SIZE AST_VIDMEM_SIZE_8M

struct ast_vbios_stdtable {
	u8 misc;
	u8 seq[4];
	u8 crtc[25];
	u8 ar[20];
	u8 gr[9];
};

struct ast_vbios_enhtable {
	u32 ht;
	u32 hde;
	u32 hfp;
	u32 hsync;
	u32 vt;
	u32 vde;
	u32 vfp;
	u32 vsync;
	u32 dclk_index;
	u32 flags;
	u32 refresh_rate;
	u32 refresh_rate_index;
	u32 mode_id;
};

struct ast_vbios_dclk_info {
	u8 param1;
	u8 param2;
	u8 param3;
};

struct ast_vbios_mode_info {
	const struct ast_vbios_stdtable *std_table;
	const struct ast_vbios_enhtable *enh_table;
};

struct ast_crtc_state {
	struct drm_crtc_state base;

	/* Last known format of primary plane */
	const struct drm_format_info *format;

	struct ast_vbios_mode_info vbios_mode_info;
};

#define to_ast_crtc_state(state) container_of(state, struct ast_crtc_state, base)

int ast_mode_config_init(struct ast_private *ast);

#define AST_MM_ALIGN_SHIFT 4
#define AST_MM_ALIGN_MASK ((1 << AST_MM_ALIGN_SHIFT) - 1)

#define AST_DP_POWER_ON			true
#define AST_DP_POWER_OFF			false

int ast_mm_init(struct ast_private *ast);

/* ast post */
void ast_enable_vga(struct drm_device *dev);
void ast_enable_mmio(struct drm_device *dev);
bool ast_is_vga_enabled(struct drm_device *dev);
void ast_post_gpu(struct drm_device *dev);
u32 ast_mindwm(struct ast_private *ast, u32 r);
void ast_moutdwm(struct ast_private *ast, u32 r, u32 v);
void ast_patch_ahb_2500(struct ast_private *ast);

/* ast_i2c.c */
struct ast_i2c_chan *ast_i2c_create(struct drm_device *dev);

/* aspeed DP */
int ast_astdp_read_edid(struct drm_device *dev, u8 *ediddata);
void ast_dp_launch(struct drm_device *dev);
void ast_dp_power_on_off(struct drm_device *dev, bool no);
void ast_dp_set_on_off(struct drm_device *dev, bool no);
void ast_dp_set_mode(struct drm_crtc *crtc, struct ast_vbios_mode_info *vbios_mode);
int ast_vga_read_edid(struct drm_device *dev, u8 *ediddata);
#endif
